/*
************************************************************************
*	Multi-alphabet Arithmetic Coding Based on Generalized Virtual Sliding Window
*	Author: Evgeny Belyaev
*	Web Page: http://eugeniy-belyaev.narod.ru/
*	Technical University of Denmark, 23/06/2017
*	This code can be used for research purpose only (no commercial usage)
*	If you use this code please refer the following paper:
E.Belyaev, S.Forchhammer, Kai Liu, An adaptive multi-alphabet arithmetic
coding based on generalized virtual sliding window // IEEE Signal Processing
Letters, vol.24, Is.7, pp.1034-1038, 2017.
************************************************************************
*/


#include <stdlib.h>
#include <stdio.h>
#include "ac.h"

/*
 ************************************************************************
 *    Initializes a given context with equal probabilities
 ************************************************************************
 */
void init_context 
(
	ContextTypePtr ctx
)
{
  int init_state = VSW_D/ALPHABET_SIZE;
  for (int i=0;i<ALPHABET_SIZE;i++)
  {
	ctx->state[i] = init_state;
  }
  ctx->cum_freq[0]=0;
  for (int i=1;i<=ALPHABET_SIZE;i++)
  {
	  ctx->cum_freq[i] = ctx->cum_freq[i-1]+ctx->state[i-1];
  }
}

/*
 ************************************************************************
 *    Allocates memory for the EncodingEnvironment struct
 ************************************************************************
 */
EncodingEnvironmentPtr create_encoding_environment()
{
  EncodingEnvironmentPtr eep;
  eep = (EncodingEnvironmentPtr)malloc(sizeof(EncodingEnvironment));
  return eep;
}

/*
 ************************************************************************
 *    Frees memory of the EncodingEnvironment struct
 ************************************************************************
 */
void delete_encoding_environment(EncodingEnvironmentPtr eep)
{
  if (eep != NULL)
  {
    free(eep);
  }
}

/*
 ************************************************************************
 *    Initializes the EncodingEnvironment for the arithmetic coder
 ************************************************************************
 */
void start_encoding(EncodingEnvironmentPtr eep,
                            unsigned char *code_buffer,
                            int *code_len )
{
  eep->Elow = 0;
  eep->Erange = HALF-2;
  eep->Ebits_to_follow = 0;
  eep->Ecodestrm = code_buffer;
  eep->Ecodestrm_len = (unsigned int*)code_len;
}

#define UINT32P(x)  (*((unsigned int*)(x)))
#define UCHARP(x) ((unsigned char*)(x))
#define Put1Bit(stream, bitptr, x)                  			\
{                                                       		\
   UINT32P(&(UCHARP(stream)[((bitptr)+7)>>3]) ) = 0;     		\
   if((x)!=0) UINT32P(&(UCHARP(stream)[(bitptr)>>3])) |=       	\
                                (unsigned int)(1L << ((bitptr)&0x7));	\
   (bitptr)++;                                            		\
}																\

/*
 ************************************************************************
 *    Terminates the arithmetic codeword, writes stop bit and stuffing bytes (if any)
 ************************************************************************
 */
void done_encoding(EncodingEnvironmentPtr eep)
{
  if((eep->Elow >> (BITS_IN_REGISTER-1)) & 1)
  {
	Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 1); 
	for (int iii = 0; iii<eep->Ebits_to_follow; iii++)
	{
		Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 0);
	}
	eep->Ebits_to_follow = 0;
  }
  else
  {
	Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 0);
	for (int iii = 0; iii<eep->Ebits_to_follow; iii++)
	{
		Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 1);
	}
	eep->Ebits_to_follow = 0;
  }
  
  Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, (eep->Elow >> (BITS_IN_REGISTER-2))&1);
  Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 1);
  *eep->Ecodestrm_len = (*eep->Ecodestrm_len+7) & ~7;
}

/*
 ************************************************************************
 *    Arithmetic encoding of one symbol utilizing
 *    the probability estimate by generalized virtual sliding window
 ************************************************************************
 */

void encode_symbol
(
 EncodingEnvironmentPtr eep, 
 signed int symbol, 
 ContextTypePtr ct
 )
{
  register unsigned int range = eep->Erange;
  register unsigned int low = eep->Elow;
  long Quant = (range-QUARTER)>>(QTRDIVFOUR);
  
  low = low + ct->cum_freq [symbol]+(Quant*(ct->cum_freq [symbol]>>MULT_PRECISE));
  range = ct->state[symbol]+(Quant*(ct->state [symbol]>>MULT_PRECISE));
  
  long freq_win = 0;
  for (int i=0;i<ALPHABET_SIZE;i++)
  {
	  ct->state[i] -= (ct->state[i] + VSW_HALF) >> (VSW_WIN_LEN);
	  freq_win += ct->state[i];
  }
  ct->state[symbol] = VSW_D-freq_win+ct->state[symbol];
  ct->cum_freq[0]=0;
  for (int i=1;i<=ALPHABET_SIZE;i++)
  {
	ct->cum_freq[i] = ct->cum_freq[i-1]+ct->state[i-1];
  }

  // renormalisation 
  while (range < QUARTER)
  {
    if (low >= HALF)
    {
		Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 1);
		for (int iii = 0; iii<eep->Ebits_to_follow; iii++)
		{
			Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 0);
		}
		eep->Ebits_to_follow = 0;
		low -= HALF;
    }
    else 
      if (low < QUARTER)
      {
		  Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 0); 
		  for (int iii = 0; iii<eep->Ebits_to_follow; iii++)
		  {
			  Put1Bit(eep->Ecodestrm, *eep->Ecodestrm_len, 1);
		  }
		  eep->Ebits_to_follow = 0;
      }
      else
      {
        eep->Ebits_to_follow++;
        low -= QUARTER;
      }
    low <<= 1;
    range <<= 1;
  }
  eep->Erange = range;
  eep->Elow = low;
}


