/*
************************************************************************
*	Multi-alphabet Arithmetic Coding Based on Generalized Virtual Sliding Window
*	Author: Evgeny Belyaev
*	Web Page: http://eugeniy-belyaev.narod.ru/
*	Technical University of Denmark, 23/06/2017
*	This code can be used for research purpose only (no commercial usage)
*	If you use this code please refer the following paper:
E.Belyaev, S.Forchhammer, Kai Liu, An adaptive multi-alphabet arithmetic
coding based on generalized virtual sliding window // IEEE Signal Processing
Letters, vol.24, Is.7, pp.1034-1038, 2017.
************************************************************************
*/


#include <stdlib.h>
#include <stdio.h>
#include "ac.h"

/*
 ************************************************************************
 *    Allocates memory for the DecodingEnvironment struct
 ************************************************************************
 */
DecodingEnvironmentPtr create_decoding_environment()
{
  DecodingEnvironmentPtr dep;
  dep = (DecodingEnvironmentPtr)malloc(sizeof(DecodingEnvironment));
  return dep;
}


/*
 ***********************************************************************
 *    Frees memory of the DecodingEnvironment struct
 ***********************************************************************
 */
void delete_decoding_environment(DecodingEnvironmentPtr dep)
{
  free(dep);
}

#define UINT32P(x)  (*((unsigned int*)(x)))
#define UCHARP(x) ((unsigned char*)(x))
#define Get1Bit(stream, bitptr, x)                      \
{                                                       \
   x   = UINT32P(&(UCHARP(stream)[(bitptr)>>3]));	\
   x >>= (bitptr) & 0x7 ;                               \
   x  &= 1;                                             \
   (bitptr)++;                                          \
}														\

/*
 ************************************************************************
 *    Initializes the DecodingEnvironment for the arithmetic coder
 ************************************************************************
 */
void start_decoding(DecodingEnvironmentPtr dep, unsigned char *cpixcode,
                            int firstbyte, int *cpixcode_len )
{
  register unsigned int bbit = 0;
 
  int value = 0;
  int i;

  dep->Dcodestrm = cpixcode;
  dep->Dcodestrm_len = (unsigned int*)cpixcode_len;
 
  dep->Dvalue = 0;
  for ( i = 1; i <= BITS_IN_REGISTER; i++)
  {
	Get1Bit(dep->Dcodestrm, *dep->Dcodestrm_len, bbit);
    dep->Dvalue = 2 * dep->Dvalue + bbit;
  }
  dep->Dlow = 0;
  dep->Drange = HALF-2;
}

/*
************************************************************************
*    Arithmetic decoding of one symbol utilizing
*    the probability estimate by generalized virtual sliding window
************************************************************************
*/
unsigned int decode_symbol
(
 DecodingEnvironmentPtr dep, 
 ContextTypePtr ct
)
{
  int bbit;
  register unsigned int low = dep->Dlow;
  register unsigned int value = dep->Dvalue;
  register unsigned int range = dep->Drange;
  unsigned int symbol=0;
  long Quant = (range-QUARTER)>>(QTRDIVFOUR);

  for (int i=0;i<ALPHABET_SIZE;i++)
  {
	long tmp = ct->cum_freq [i]+Quant*(ct->cum_freq [i]>>MULT_PRECISE);
    if (tmp > (value - low))
	{
		break;		
	}
	symbol = i;
  }
	  
  low = low + ct->cum_freq [symbol]+(Quant*(ct->cum_freq [symbol]>>MULT_PRECISE));
  range = ct->state[symbol]+(Quant*(ct->state [symbol]>>MULT_PRECISE));
  
  long freq_win = 0;
  for (int i=0;i<ALPHABET_SIZE;i++)
  {
	  ct->state[i] -= (ct->state[i] + VSW_HALF) >> (VSW_WIN_LEN);;
	  freq_win += ct->state[i];
  }
  ct->state[symbol] = VSW_D-freq_win+ct->state[symbol];

  ct->cum_freq[0]=0;
  for (int i=1;i<=ALPHABET_SIZE;i++)
  {
	ct->cum_freq[i] = ct->cum_freq[i-1]+ct->state[i-1];
  }

  while (range < QUARTER)
  {
 	if (low >= HALF)
    {
        low -= HALF;
		value -= HALF;
    }
    else 
      if (low < QUARTER)
      {
      }
      else
      {
        low -= QUARTER;
		value -= QUARTER;
      }
    range <<= 1;
	low <<= 1;
	Get1Bit(dep->Dcodestrm, *dep->Dcodestrm_len, bbit);
    value = (value << 1) | bbit;
  }

  dep->Drange = range;
  dep->Dvalue = value;
  dep->Dlow = low;
 
  return(symbol);
}


