/*
************************************************************************
*	Multi-alphabet Arithmetic Coding Based on Generalized Virtual Sliding Window
*	Author: Evgeny Belyaev
*	Web Page: http://eugeniy-belyaev.narod.ru/
*	Technical University of Denmark, 23/06/2017
*	This code can be used for research purpose only (no commercial usage)
*	If you use this code please refer the following paper:
E.Belyaev, S.Forchhammer, Kai Liu, An adaptive multi-alphabet arithmetic
coding based on generalized virtual sliding window // IEEE Signal Processing
Letters, vol.24, Is.7, pp.1034-1038, 2017.
************************************************************************
*/


#ifndef ACCODER_H
#define ACCODER_H

//Alphabet size
#define ALPHABET_SIZE_LOG2 8
#define ALPHABET_SIZE (1<<ALPHABET_SIZE_LOG2)
//Generalized Virtual Sliding Sindow Size LOG2 7..11
#define VSW_WIN_LEN 9
//Multiplication precision 0..8
#define MULT_PRECISE 8

#define VSW_D_LOG2 (2*VSW_WIN_LEN+8)
#define VSW_D  (1<<(VSW_D_LOG2))
#define VSW_HALF ((1<<(VSW_WIN_LEN-1)))
#define BITS_IN_REGISTER (VSW_D_LOG2+2)
#define QUARTER (1 << (BITS_IN_REGISTER-2))
#define HALF (1 << (BITS_IN_REGISTER-1))
#define QTRDIVFOUR (BITS_IN_REGISTER-2-MULT_PRECISE)

// context struct
typedef struct
{
  long state[ALPHABET_SIZE];			
  unsigned long cum_freq[ALPHABET_SIZE+1];
} ContextType;
typedef ContextType *ContextTypePtr;

// struct and functions of arithmetic encoding engine
typedef struct
{
  unsigned int  Elow, Erange;
  unsigned int  Ebits_to_follow;
  unsigned char *Ecodestrm;
  unsigned int  *Ecodestrm_len;
} EncodingEnvironment;
typedef EncodingEnvironment *EncodingEnvironmentPtr;
EncodingEnvironmentPtr create_encoding_environment();
void init_context(ContextTypePtr ctx);
void encode_symbol(EncodingEnvironmentPtr eep, signed int symbol, ContextTypePtr ct);
void delete_encoding_environment(EncodingEnvironmentPtr eep);
void start_encoding(EncodingEnvironmentPtr eep, unsigned char *code_buffer, int *code_len);
void done_encoding(EncodingEnvironmentPtr eep);

// struct and functions of arithmetic dencoding engine
typedef struct
{
  unsigned int    Dlow, Drange;
  unsigned int    Dvalue;
  int             Dbits_to_go;
  unsigned char   *Dcodestrm;
  unsigned int    *Dcodestrm_len;
} DecodingEnvironment;
typedef DecodingEnvironment *DecodingEnvironmentPtr;
DecodingEnvironmentPtr create_decoding_environment();
void delete_decoding_environment(DecodingEnvironmentPtr dep);
void start_decoding(DecodingEnvironmentPtr dep, unsigned char *cpixcode,int firstbyte, int *cpixcode_len );
unsigned int decode_symbol(DecodingEnvironmentPtr dep, ContextTypePtr ct);

#endif // !ACCODER_H