/*
************************************************************************
*	Multi-alphabet Arithmetic Coding Based on Generalized Virtual Sliding Window
*	Author: Evgeny Belyaev
*	Web Page: http://eugeniy-belyaev.narod.ru/
*	Technical University of Denmark, 23/06/2017
*	This code can be used for research purpose only (no commercial usage)
*	If you use this code please refer the following paper:
E.Belyaev, S.Forchhammer, Kai Liu, An adaptive multi-alphabet arithmetic
coding based on generalized virtual sliding window // IEEE Signal Processing
Letters, vol.24, Is.7, pp.1034-1038, 2017.
************************************************************************
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ac.h"

void Encode(char *fname1, char *fname2);
void Decode(char *fname1, char *fname2);

void DecodeBuffer(unsigned char* code_buffer, int numof_encsymbol, unsigned char* dec_buffer);
unsigned int EncodeBuffer(unsigned char *InputBuffer,	unsigned int input_len,unsigned char** compress);

int FileCompare(char *fname1, char *fname2);

ContextType ctx0;

void main(int argc, char* argv[])
{
   char dec_name[512];
   int len=0;

   memset(dec_name,0,512);
   for (int i=0;i<strlen(argv[1]);i++,len++)
   {
	dec_name[i] = argv[1][i];
   }
   dec_name[len++]='_';
   dec_name[len++]='d';
   dec_name[len++]='e';
   dec_name[len++]='c';
   dec_name[len++]='.';
   dec_name[len++]=argv[1][strlen(argv[1])-3];
   dec_name[len++]=argv[1][strlen(argv[1])-2];
   dec_name[len++]=argv[1][strlen(argv[1])-1];

   Encode(argv[1], "stream.dat");
   Decode("stream.dat", dec_name);
   printf("differences = %d\n", FileCompare(argv[1],dec_name));
}


void Encode
(
	char *fname1,
	char *fname2
)
{
	unsigned char* InputBuffer;
	unsigned char* OutputBuffer;
	unsigned int inp_buffer_len;
	int enc_len;
	unsigned int numof_encsymbol;


	FILE* fp;
	fp = fopen(fname1, "rb");
	fseek(fp, 0, SEEK_END);
	inp_buffer_len = ftell(fp);
	fclose(fp);

	fp = fopen(fname1, "rb");
	InputBuffer = (unsigned char*)malloc(inp_buffer_len);

	numof_encsymbol = inp_buffer_len;
	fread(InputBuffer, 1, inp_buffer_len, fp);
	fclose(fp);

	enc_len = EncodeBuffer(InputBuffer, numof_encsymbol, &OutputBuffer);

	fp = fopen(fname2, "wb");
	fwrite(&numof_encsymbol, 4, 1, fp);
	fwrite(&enc_len, 4, 1, fp);
	fwrite(OutputBuffer, enc_len / 8 + 1, 1, fp);
	fclose(fp);

	if (OutputBuffer != NULL) free(OutputBuffer);
	if (InputBuffer != NULL)  free(InputBuffer);
}

void Decode
(
	char *fname1,
	char *fname2
)
{
	unsigned char* code_buffer, *dec_buffer;
	int code_len = 0;
	int numof_encsymbol = 0;
	int len = 0;


	FILE* fp;
	fp = fopen(fname1, "rb");
	fread(&numof_encsymbol, 4, 1, fp);
	fread(&len, 4, 1, fp);
	printf("\nlen=%d\n", len / 8);

	code_buffer = (unsigned char*)malloc(len);
	dec_buffer = (unsigned char*)malloc(numof_encsymbol);

	fread(code_buffer, len, 1, fp);
	fclose(fp);

	DecodeBuffer(code_buffer, numof_encsymbol, dec_buffer);

	fp = fopen(fname2, "wb");
	fwrite(dec_buffer, 1, numof_encsymbol, fp);
	fclose(fp);
	free(code_buffer);
	free(dec_buffer);
}


unsigned int EncodeBuffer
(
	unsigned char *InputBuffer, 
	unsigned int input_len, 
	unsigned char** compress
)
{
  EncodingEnvironmentPtr eep;
  unsigned char* code_buffer;
  int code_len=0;
  unsigned int freq_all = 0;

  code_buffer  = (unsigned char*)malloc((int)(input_len*1.5));
  memset(code_buffer, 1, sizeof(code_buffer) );
  init_context(&ctx0);
  eep = create_encoding_environment(); 
  start_encoding(eep, code_buffer, &code_len ); 

  for (int i = 0; i < input_len; i++)
  {
	  int symb = InputBuffer[i];
	  encode_symbol(eep, symb, &ctx0);
  }

  encode_symbol(eep, 32, &ctx0);
  encode_symbol(eep, 32, &ctx0);
  encode_symbol(eep, 32, &ctx0);
  encode_symbol(eep, 32, &ctx0);

  done_encoding(eep); 
  delete_encoding_environment(eep);

  *compress = code_buffer;
   
  double R = (double)code_len/(double)input_len;
  FILE *fp_stat = fopen("freq_stat.txt","at");
  fprintf(fp_stat,"%g\n",R);
  fclose(fp_stat);

  return code_len + 1;
}

void DecodeBuffer
(
	unsigned char* code_buffer, 
	int numof_encsymbol, 
	unsigned char* dec_buffer
)
{

  DecodingEnvironmentPtr dep;
  int code_len=0;
  
  init_context(&ctx0);
  dep = create_decoding_environment();
  start_decoding(dep, code_buffer, 0, &code_len );

  for(int len = 0; len<numof_encsymbol; len++ )
  {
	  int symb = decode_symbol(dep, &ctx0);
    dec_buffer[len] = symb;
  }
  delete_decoding_environment(dep);
}

int FileCompare
(
	char *fname1, 
	char *fname2
)
{
  FILE* fp1, *fp2;
  char symb1, symb2;
  int not_equ = 0;
  int numof_encsymbol;
  int i;
  
  fp1 = fopen(fname1, "rt");
  fp2 = fopen(fname2, "rt");
  numof_encsymbol=0;
  
  for( i = 0; (!feof(fp1)) && (!feof(fp2)); i++ )
  {
	fscanf(fp1, "%c", &symb1);
	fscanf(fp2, "%c", &symb2);
	numof_encsymbol++;
	if( symb1 != symb2 )
	{
	  printf( "%d: %c %c\n", i, symb1, symb2 );
	  not_equ++;
	}
  }

  if( i != numof_encsymbol )
  {
	printf( "files have different length\n" );
  }

  fclose(fp1);
  fclose(fp2);

  return not_equ;
}

